module net.BurtonRadons.dedit.mainCommands;

import net.BurtonRadons.dedit.main;

/** Provides the standard command set. */
class MainCommandSet
{
    this ()
    {
        alias View.addCommand e;

        e ("CharLeft", "", &CharLeft,
            "Move the cursor one character to the left.",
            "Moves the cursor one character to the left.  If the cursor is at the start "
            "of a line, it moves to the end of the previous line.  If the cursor is at the "
            "beginning of the document, it does not move.  This adjusts scrolling as "
            "needed.  This clears the selection.");
        e ("CharRight", "", &CharRight,
            "Move the cursor one character to the right.",
            "Moves the cursor one character to the right.  If the cursor is at the end "
            "of a line, it moves to the beginning of the next line.  If the cursor is at the "
            "end of the document, it does not move.  This adjusts scrolling as "
            "needed.  This clears the selection.");
        e ("LineDown", "", &LineDown,
            "Move the cursor down one line.",
            "Moves the cursor down one line.  This changes the new horizontal "
            "position of the cursor to be as close visually to the previous pixel "
            "mark, which is assigned by all cursor movements except for those "
            "that change the line.  For example, if you use this command and "
            "the cursor lands on a shorter line, then the cursor will go to the end "
            "of the line, but if you go back to the previous line, then it will go back "
            "to the original horizontal position.  This clears the selection.");
        e ("LineUp", "", &LineUp,
            "Move the cursor up one line.",
            "Moves the cursor up one line.  This changes the new horizontal "
            "position of the cursor to be as close visually to the previous pixel "
            "mark, which is assigned by all cursor movements except for those "
            "that change the line.  For example, if you use this command and "
            "the cursor lands on a shorter line, then the cursor will go to the end "
            "of the line, but if you go back to the previous line, then it will go back "
            "to the original horizontal position.  This clears the selection.");
        e ("HomeOrLineStart", "", &HomeOrLineStart,
            "Go to the beginning of the line or the first non-whitespace character.",
            "This toggles the horizontal positioning of the cursor between the beginning "
            "of the line and the first non-whitespace character if there is one.  If there is "
            "not, it uses the end of the line.  This clears the selection.");
        e ("End", "", &End,
            "Move the cursor to the end of the line.",
            "Move the cursor to the end of the line.  This clears the selection.");
        e ("PageUp", "", &PageUp,
            "Move the cursor up a page and scroll the same distance.",
            "Moves the cursor up a page.  This changes the new horizontal "
            "position of the cursor to be as close visually to the previous pixel "
            "mark, which is assigned by all cursor movements except for those "
            "that change the line.  For example, if you use this command and "
            "the cursor lands on a shorter line, then the cursor will go to the end "
            "of the line, but if you go back to the previous line, then it will go back "
            "to the original horizontal position.  This clears the selection.");
        e ("PageDown", "", &PageDown,
            "Move the cursor down a page and scroll the same distance.",
            "Moves the cursor down a page.  This changes the new horizontal "
            "position of the cursor to be as close visually to the previous pixel "
            "mark, which is assigned by all cursor movements except for those "
            "that change the line.  For example, if you use this command and "
            "the cursor lands on a shorter line, then the cursor will go to the end "
            "of the line, but if you go back to the previous line, then it will go back "
            "to the original horizontal position.  This clears the selection.");
        e ("DeleteSelectionOrPrev", "", &DeleteSelectionOrPrev,
            "Delete the current selection or the previous character.",
            "This deletes the current selection if there is one or the previous "
            "character if there isn't.  In the latter case, if the cursor is at the "
            "beginning of the line it moves the cursor and the current line "
            "to the end of the previous line.  If the cursor is at the beginning "
            "of the document, nothing is done in the latter case.  If the cursor "
            "is at the start of whitespace in the line, then it deletes to the next "
            "lower tab level as determined by scanning lines prior to this one.");
        e ("DeleteSelectionOrNext", "", &DeleteSelectionOrNext,
            "Delete the current selection or the next character.",
            "This deletes the current selection if there is one or the next "
            "character if there isn't.  In the latter case, if the cursor is at the "
            "end of the line it moves the next line to the end of this line.  If "
            "the cursor is at the end of the document, nothing is done in the "
            "latter case.");
        e ("DocumentStart", "", &DocumentStart,
            "Move to the beginning of the document.",
            "Move the cursor to the beginning of the document.  This clears the "
            "selection.");
        e ("DocumentStartExtend", "", &DocumentStartExtend,
            "Move to the beginning of the document, extending the selection.",
            "Move the cursor to the beginning of the document.  This extends the "
            "selection.");
        e ("DocumentEnd", "", &DocumentEnd,
            "Move to the end of the document.",
            "Move the cursor to the end of the document.  This clears the selection.");
        e ("DocumentEndExtend", "", &DocumentEndExtend,
            "Move to the end of the document, extending selection.",
            "Move the cursor to the end of the document.  This extends the selection.");
        e ("Return", "", &Return,
            "Insert a new line, splitting this line at the cursor.",
            "Create a new line and put everything after the cursor at the end of it.  "
            "This will automatically indent the line according to the syntax "
            "highlighter's settings.  This clears the selection.");
        e ("DeleteLine", "", &DeleteLine,
            "Delete the current line.",
            "Delete the current line and move the cursor to the beginning of the "
            "next line.  If this is the last line, it moves the cursor to the beginning of "
            "the previous line.  This clears the selection.");
        e ("CharRightExtend", "", &CharRightExtend,
            "Move cursor to the right, extending selection.",
            "Moves the cursor one character to the right.  If the cursor is at the end of the line, "
            "it is moved to the beginning of the next line.  If the cursor is at the end of the "
            "document, no changes are made.  This extends the selection.");            
        e ("CharLeftExtend", "", &CharLeftExtend,
            "Move cursor to the left, extending selection.",
            "Moves the cursor one character to the left.  If the cursor is at the beginning of the line, "
            "it is moved to the end of the previous line.  If the cursor is at the beginning of the "
            "document, no changes are made.  This extends the selection.");
        e ("LineDownExtend", "", &LineDownExtend,
            "Move cursor down a line, extending selection.",
            "Moves the cursor down a line.  If the cursor is at the last line in the document, it is "
            "moved to the end of the line.  This extends the selection.");
        e ("LineUpExtend", "", &LineUpExtend,
            "Move cursor up a line, extending selection.",
            "Moves the cursor up a line.  If the cursor is at the first line in the document, it is "
            "moved to the beginning of the line.  This extends the selection.");
        e ("HomeOrLineStartExtend", "", &HomeOrLineStartExtend,
            "Extend the selection to the beginning of the line.",
            "This toggles the horizontal positioning of the cursor between the beginning "
            "of the line and the first non-whitespace character if there is one.  If there is "
            "not, it uses the end of the line.  This extends the selection.");
        e ("EndExtend", "", &EndExtend,
            "Extend the selection to the end of the line.",
            "Move the cursor to the end of the line.  This extends the selection.");
        e ("PageUpExtend", "", &PageUpExtend,
            "Extend the selection up a page.",
            "Moves the cursor up a page.  This changes the new horizontal "
            "position of the cursor to be as close visually to the previous pixel "
            "mark, which is assigned by all cursor movements except for those "
            "that change the line.  For example, if you use this command and "
            "the cursor lands on a shorter line, then the cursor will go to the end "
            "of the line, but if you go back to the previous line, then it will go back "
            "to the original horizontal position.  This extends the selection.");
        e ("PageDownExtend", "", &PageDownExtend,
            "Extend the selection down a page.",
            "Moves the cursor up a page.  This changes the new horizontal "
            "position of the cursor to be as close visually to the previous pixel "
            "mark, which is assigned by all cursor movements except for those "
            "that change the line.  For example, if you use this command and "
            "the cursor lands on a shorter line, then the cursor will go to the end "
            "of the line, but if you go back to the previous line, then it will go back "
            "to the original horizontal position.  This extends the selection.");
        e ("DeleteSelection", "", &DeleteSelection,
            "Delete the current selection.",
            "Deletes the current selection.  If there is no selection, this has no effect.");
        e ("DeleteNext", "", &DeleteNext,
            "Delete the next character.",
            "Delete the next character from the cursor.  If the cursor is at the end of the "
            "line, it appends the contents of the next line to the current line.  If the cursor "
            "is at the end of the document, it has no effect.  This clears the selection.");
        e ("DeletePrev", "", &DeletePrev,
            "Delete the previous character.",
            "Delete the character before the cursor, and move the cursor to the left.  If "
            "the cursor is at the beginning of the line, the current line is appended to the "
            "end of the previous line and the cursor is put at that point as well.  If the "
            "cursor is at the beginning of the document, it has no effect.  If the only "
            "text to the left of the cursor is whitespace, this deletes up to the next "
            "indentation level as determined by scanning the lines above.  "
            "This clears the selection.");
        e ("SelectSyntaxHighlighter", ">", &SelectSyntaxHighlighter,
            "Present a menu selecting the syntax highlighter.",
            "Shows a drop-down menu with which you can select the syntax highlighter "
            "for the current document.");
        e ("SelectAll", "", &SelectAll,
            "Select the entire document.",
            "Select the entire document and move the cursor to the beginning of the "
            "document.");
        e ("Copy", "", &Copy,
            "Copy the selection onto the clipboard.",
            "Copies the current selection onto the clipboard.  If there is no selection, "
            "there is no effect.");
        e ("Cut", "", &Cut,
            "Move the selection onto the clipboard, deleting it from the document.",
            "Move the current selection onto the clipboard, deleting it from the document.  "
            "If there is no selection, there is no effect.");
        e ("Paste", "", &Paste,
            "Paste the clipboard into the document, deleting the selection.",
            "Pastes the clipboard into the document.  If there is a selection, its contents "
            "are deleted.  This effect persists even if there is no text on the clipboard.");
        e ("Undo", "u", &Undo,
            "Undoes the previous action.",
            "Undoes the previous action in the current document.  If this is at the beginning "
            "of the undo buffer, no effect occurs.  Only changes to the text of the document are "
            "recorded; cursor changes, preferences, or anything else are not recorded.");
        e ("Redo", "U", &Redo,
            "Redoes the action most recently undone.",
            "Redoes the action most recently undone.  The current mark in the undo buffer is "
            "reset to the end if the text is modified; hence undoing, changing the text, and redoing "
            "will have no effect.");
        e ("Indent", "", &Indent,
            "Increase indentation.",
            "Increase indentation if at the beginning of a line, or insert a tab if in the middle or "
            "end of it.  If there is a selection, then all lines in the selection are indented from the "
            "beginning of the line.  Indentation behaviour is controlled through the options window.");
        e ("Dedent", "", &Dedent,
            "Decrease indentation.",
            "Decrease indentation if at the beginning of a line, or remove a tab if in the middle or "
            "end of it.  If there is a selection, then all lines in the selection are dedented from the "
            "beginning of the line.  If this would delete more spaces than there are, then it deletes "
            "all possible spaces, so dedenting a block of text with variable indentation multiple times "
            "will eventually result in a straight block of text.  Indentation behaviour is controlled "
            "through the options menu.");
        e ("Exit", "", &Exit,
            "Close the program.",
            "Close the program.  If the project is unnamed, it will ask to save it, and give an option "
            "to cancel the exit.  If there are unsaved documents, it will ask to save them, and give an "
            "option to cancel the exit.  Then if not canceled it will close the program and return zero "
            "from it.");
        e ("ScrollDown", "", &ScrollDown,
            "Scroll down a line.",
            "Scrolls the view of the current document down a line.  This will not scroll to show anything "
            "beyond the last line of the document.");
        e ("ScrollUp", "", &ScrollUp,
            "Scroll up a line.",
            "Scrolls the view of the current document up a line.  This will not scroll to show anything "
            "before the first line of the document.");
        e ("ScrollCenter", "", &ScrollCenter,
            "Center scrolling on the cursor.",
            "Scrolls the view of the current document so that the cursor is centered in it.  This will not "
            "scroll to show anything before the first line of the document or after the last line.");
        e ("SegmentLeft", "", &SegmentLeft,
            "Skip a symbol, a length of whitespace, or an identifier to the left.",
            "If the cursor is at the beginning of the line, it moves to the end of the previous line "
            "and finishes processing.  Otherwise it looks at the character just to the left of the "
            "cursor.  If this is whitespace, an identifier, or a number, it skips it to the left up until the "
            "beginning of the line at maximum.  If this is a symbol, it moves one character "
            "to the left.  If this is a string or comment, then it skips a word, whitespace, or "
            "a symbol as appropriate.  This is a variation on the word-left command of "
            "most editors but is usually more useful for macros.  This clears the selection.");
        e ("SegmentRight", "", &SegmentRight,
            "Skip a symbol, a length of whitespace, or an identifier to the right.",
            "If the cursor is at the end of the line, it moves to the beginning of the next line "
            "and finishes processing.  Otherwise it looks at the character under the "
            "cursor.  If this is whitespace, an identifier, or a number, it skips it to the right up until the "
            "end of the line at maximum.  If this is a symbol, it moves one character "
            "to the right.  If this is a string or comment, then it skips a word, whitespace, or "
            "a symbol as appropriate.  This is a variation on the word-right command of "
            "most editors but is usually more useful for macros.  This clears the selection.");
        e ("SegmentLeftExtend", "", &SegmentLeftExtend,
            "Skip a symbol, a length of whitespace, or an identifier to the left.",
            "If the cursor is at the beginning of the line, it moves to the end of the previous line "
            "and finishes processing.  Otherwise it looks at the character just to the left of the "
            "cursor.  If this is whitespace, an identifier, or a number, it skips it to the left up until the "
            "beginning of the line at maximum.  If this is a symbol, it moves one character "
            "to the left.  If this is a string or comment, then it skips a word, whitespace, or "
            "a symbol as appropriate.  This is a variation on the word-left command of "
            "most editors but is usually more useful for macros.  This extends the selection.");
        e ("SegmentRightExtend", "", &SegmentRightExtend,
            "Skip a symbol, a length of whitespace, or an identifier to the right.",
            "If the cursor is at the end of the line, it moves to the beginning of the next line "
            "and finishes processing.  Otherwise it looks at the character under the "
            "cursor.  If this is whitespace, an identifier, or a number, it skips it to the right up until the "
            "end of the line at maximum.  If this is a symbol, it moves one character "
            "to the right.  If this is a string or comment, then it skips a word, whitespace, or "
            "a symbol as appropriate.  This is a variation on the word-right command of "
            "most editors but is usually more useful for macros.  This extends the selection.");
        e ("MacroRecordToggle", "m", &MacroRecordToggle,
            "Toggle macro recording.",
            "Toggle whether macro recording is enabled.  When enabled, all commands are added "
            "to a macro script, including those commands executed through a menu or the binding "
            "dialog.  They can then be played back using MacroPlay.  There is one macro recorded "
            "at a time per project.");
        e ("MacroPlay", "", &MacroPlay,
            "Play back a previously recorded macro.",
            "Plays back a previously recorded macro.  There is a single current macro for "
            "each project.  If no macro has been recorded, there is no effect.");
    }

    static this ()
    {
        new MainCommandSet ();
    }
    
    void MacroPlay (View view)
    {
        char [] [] macro = view.macro;
        
        for (int c; c < macro.length; c ++)
        {
            char [] command = macro [c];
            
            if (command [0] == '*')
                view.runChar (command [1]);
            else
                view.run (command);
        }
    }
    
    void MacroRecordToggle (View view)
    {
        if (view.macroRecording)
        {
            view.macro = view.macroNext;
            global.statusBar.display ("Macro recording FINISHED");
        }
        else
        {
            view.macroNext = null;
            global.statusBar.display ("Macro recording BEGUN");
        }
        view.macroRecording = !view.macroRecording;        
    }
    
    void SegmentLeftBase (View view, bit extend)
    {
        with (view)
        {
            int oline = document.line;
            int ooffset = document.offset;
            char [] high = document.highs [document.line];
            char [] line = document.lines [document.line];
            
            if (document.offset == 0)
            {
                document.line = imax (0, document.line - 1);
                document.offset = document.lines [document.line].length;
            }
            else if (document.offset == 1)
                document.offset = 0;
            else
            {
                char type = high [document.offset - 1];

                /* Use word/symbol/whitespace skipping. */
                if (type == '*' || type == '\"')
                {
                    if (std.ctype.isspace (line [document.offset - 1]))
                    {
                        do document.offset --;
                        while (document.offset > 0 && std.ctype.isspace (line [document.offset - 1]));
                    }
                    else if (std.ctype.isalnum (line [document.offset - 1]) || line [document.offset - 1] == '_')
                    {
                        do document.offset --;
                        while (document.offset > 0 && (std.ctype.isalnum (line [document.offset - 1]) || line [document.offset - 1] == '_'));
                    }
                    else
                        document.offset --;
                }
                else if (type == 's') /* Skip a single character. */
                    document.offset --;
                else /* Otherwise skip all characters of this type. */
                {
                    do document.offset --;
                    while (document.offset > 0 && high [document.offset - 1] == type);
                }
            }
            
            if (extend)
                document.selFollowCursor (oline, ooffset, document.line, document.offset);
            else
                document.selClear ();
            view.caretChange ();
        }
    }

    void SegmentRightBase (View view, bit extend)
    {
        with (view)
        {
            int oline = document.line;
            int ooffset = document.offset;
            char [] high = document.highs [document.line];
            char [] line = document.lines [document.line];
            
            if (document.offset == line.length)
            {
                if (document.line == document.lines.length - 1)
                    document.offset = line.length;
                else
                {
                    document.line ++;
                    document.offset = 0;
                }
            }
            else if (document.offset == line.length - 1)
                document.offset = line.length;
            else
            {
                char type = high [document.offset];

                /* Use word/symbol/whitespace skipping. */
                if (type == '*' || type == '\"')
                {
                    if (std.ctype.isspace (line [document.offset]))
                    {
                        do document.offset ++;
                        while (document.offset < line.length && std.ctype.isspace (line [document.offset]));
                    }
                    else if (std.ctype.isalnum (line [document.offset]) || line [document.offset] == '_')
                    {
                        do document.offset ++;
                        while (document.offset < line.length && (std.ctype.isalnum (line [document.offset]) || line [document.offset] == '_'));
                    }
                    else
                        document.offset ++;
                }
                else if (type == 's') /* Skip a single character. */
                    document.offset ++;
                else /* Otherwise skip all characters of this type. */
                {
                    do document.offset ++;
                    while (document.offset < line.length && high [document.offset] == type);
                }
            }
            
            if (extend)
                document.selFollowCursor (oline, ooffset, document.line, document.offset);
            else
                document.selClear ();
            view.caretChange ();
        }
    }
    
    void SegmentLeft (View view)
    {
        SegmentLeftBase (view, false);
    }
    
    void SegmentLeftExtend (View view)
    {
        SegmentLeftBase (view, true);
    }
    
    void SegmentRight (View view)
    {
        SegmentRightBase (view, false);
    }
    
    void SegmentRightExtend (View view)
    {
        SegmentRightBase (view, true);
    }

    void Exit (View view)
    {
        if (view.projectName !== null)
            view.run ("ProjectSave");

        view.registry.saveString (register ~ "project", view.projectName);
        view.confirmModified ();
        global.window.quit ();
    }

    int findIndent (View view, int start)
    {
        int line;

        return findIndent (view, start, line);
    }

    /* Return the indentation at this point. */
    int findIndent (View view, int start, out int line)
    {
        TabParams p = view.document.tabParams ();
        int indent;

        for (int c = start; c >= 0; c --)
        {
            line = c;

            char [] la = view.document.lines [c];

            indent = 0;
            for (char *l = la, e = l + la.length; l < e; l ++)
            {
                if (*l == ' ')
                    indent ++;
                else if (*l == '\t')
                    indent = (indent + p.tabSize) / p.tabSize * p.tabSize;
                else if (!std.ctype.isspace (*l))
                    return indent;
            }
        }

        return 0;
    }

    int findLineStart (View view, int line)
    {
        char [] t = view.document.lines [line];

        for (int c = 0; c < t.length; c ++)
            if (!std.ctype.isspace (t [c]))
                return c;

        return t.length;
    }

    void Undo (View view)
    {
        Document d = view.document;

        d.selClear ();
        d.undo (d.line, d.offset);
        view.caretChange ();
    }

    void Redo (View view)
    {
        Document d = view.document;

        d.selClear ();
        d.redo (d.line, d.offset);
    }

    void SelectAll (View view)
    {
        with (view.document)
        {
            line = 0;
            offset = 0;
            selStartLine = selStartOffset = 0;
            selEndLine = lines.length - 1;
            selEndOffset = lines [selEndLine].length;
            view.caretChange ();
        }
    }

    void DocumentStartBase (View view, bit extend)
    {
        int oline = view.document.line, ooffset = view.document.offset;

        with (view.document)
        {
            offset = line = 0;
            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretChange ();
        }
    }

    void DocumentStart (View view) { DocumentStartBase (view, false); }
    void DocumentStartExtend (View view) { DocumentStartBase (view, true); }

    void DocumentEndBase (View view, bit extend)
    {
        int oline = view.document.line, ooffset = view.document.offset;

        with (view.document)
        {
            line = lines.length - 1;
            offset = lines [line].length;
            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretChange ();
        }
    }

    void DocumentEnd (View view) { DocumentEndBase (view, false); }
    void DocumentEndExtend (View view) { DocumentEndBase (view, true); }

    class ChangeHighlighter
    {
        View view;
        SyntaxHighlighter value;

        this (Dispatcher *d, View view, SyntaxHighlighter value)
        {
            this.view = view;
            this.value = value;
            d.add (&run);
        }

        void run (Event e)
        {
            view.document.syntaxHighlighter (value);
            view.clean ();
        }
    }

    void SelectSyntaxHighlighter (View view)
    {
        SyntaxHighlighter [] list = SyntaxHighlighter.list;
        Menu menu = view.returnMenu = new Menu (view);
        char [] [] names;

        names.length = list.length;
        for (int c; c < list.length; c ++)
            names [c] = list [c].name ();
        names.sort;

        for (int c; c < list.length; c ++)
        {
            SyntaxHighlighter high;
            char [] name = names [c];

            for (int d; d < list.length; d ++)
                if (name == list [d].name ())
                {
                    high = list [d];
                    break;
                }

            new ChangeHighlighter (menu.add (name), view, high);
        }
    }

    void DeleteLine (View view)
    {
        with (view.document)
        {
            dirtyFrom (line);
            if (lines.length > 1)
            {
                delLine (line);
                if (line >= lines.length)
                    line --;
                offset = view.findNearestX (line, view.caretX);
            }
            else
            {
                setLine (0, null);
                offset = 0;
            }

            selClear ();
            view.caretChange ();
        }
    }

    void CharLeftBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            offset --;
            if (offset < 0)
            {
                line --;
                if (line < 0)
                    line = offset = 0;
                else
                    offset = lines [line].length;
            }

            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretChange ();
        }
    }

    void CharLeft (View view) { CharLeftBase (view, false); }
    void CharLeftExtend (View view) { CharLeftBase (view, true); }

    void CharRightBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            offset ++;
            if (offset > lines [line].length)
            {
                line ++;
                if (line == lines.length)
                    line --, offset --;
                else
                    offset = 0;
            }

            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretChange ();
        }
    }

    void CharRight (View view) { CharRightBase (view, false); }
    void CharRightExtend (View view) { CharRightBase (view, true); }

    void LineDownBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            if (line == lines.length - 1)
                return;

            line ++;
            offset = view.findNearestX (line, view.caretX);
            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretInWindow ();
        }
    }

    void LineDown (View view) { LineDownBase (view, false); }
    void LineDownExtend (View view) { LineDownBase (view, true); }

    void LineUpBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            if (line == 0)
                return;

            line --;
            offset = view.findNearestX (line, view.caretX);
            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretInWindow ();
        }
    }

    void LineUp (View view) { LineUpBase (view, false); }
    void LineUpExtend (View view) { LineUpBase (view, true); }

    void HomeOrLineStartBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            char [] text = lines [line];
            int e;

            for (e = 0; e < text.length && std.ctype.isspace (text [e]); e ++)
                {}

            if (offset == e)
                offset = 0;
            else
                offset = e;

            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();

            view.caretChange ();
        }
    }

    void HomeOrLineStart (View view) { HomeOrLineStartBase (view, false); }
    void HomeOrLineStartExtend (View view) { HomeOrLineStartBase (view, true); }

    void EndBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            offset = lines [line].length;
            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretChange ();
        }
    }

    void End (View view) { EndBase (view, false); }
    void EndExtend (View view) { EndBase (view, true); }

    void PageUpBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            lineOffset -= view.pageHeight;
            line -= view.pageHeight;
            if (line < 0)
                line = offset = 0;
            else
                offset = view.findNearestX (line, view.caretX);
            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretChange ();
        }
    }

    void PageUp (View view) { PageUpBase (view, false); }
    void PageUpExtend (View view) { PageUpBase (view, true); }

    void PageDownBase (View view, bit extend)
    {
        int oline = view.document.line;
        int ooffset = view.document.offset;

        with (view.document)
        {
            lineOffset += view.pageHeight;
            line += view.pageHeight;
            if (line >= lines.length)
            {
                line = lines.length - 1;
                offset = lines [line].length;
            }
            else
                offset = view.findNearestX (line, view.caretX);
            if (extend)
                selFollowCursor (oline, ooffset, line, offset);
            else
                selClear ();
            view.caretChange ();
        }
    }

    void PageDown (View view) { PageDownBase (view, false); }
    void PageDownExtend (View view) { PageDownBase (view, true); }

    void DeletePrev (View view)
    {
        with (view.document)
        {
            if (offset)
            {
                char [] t = lines [line];
                char [] i;

                if (offset != findLineStart (view, line))
                {
                    setLine (line, t [0 .. offset - 1] ~ t [offset .. t.length]);
                    offset --;
                    view.caretChange ();
                    return;
                }

                int outline, base, indent;

                outline = line;
                base = offset;
                while (1)
                {
                    indent = findIndent (view, outline - 1, outline);
                    if (indent < base || indent == 0)
                        break;
                }

                i.length = indent;
                for (int c; c < indent; c ++)
                    i [c] = ' ';

                setLine (line, i ~ t [offset .. t.length]);
                offset = indent;
                view.caretChange ();
            }
            else if (line)
            {
                char [] t = lines [line - 1];
                char [] l = lines [line];

                setLine (line - 1, t ~ l);
                delLine (line);
                line --;
                offset = t.length;
                view.caretChange ();
            }
        }
    }

    void DeleteSelectionOrPrev (View view)
    {
        if (view.document.selExists ())
            DeleteSelection (view);
        else
            DeletePrev (view);
    }

    void DeleteNext (View view)
    {
        with (view.document)
        {
            char [] t = lines [line];

            if (offset < t.length)
                setLine (line, t [0 .. offset] ~ t [offset + 1 .. t.length]);
            else if (line < lines.length - 1)
            {
                setLine (line, t ~ lines [line + 1]);
                delLine (line + 1);
            }
            else
                return;

            view.caretChange ();
        }
    }

    void DeleteSelectionOrNext (View view)
    {
        if (view.document.selExists ())
            DeleteSelection (view);
        else
            DeleteNext (view);
    }

    void DeleteSelection (View view)
    {
        with (view.document)
        {
            if (!selExists ())
                return;
            line = selStartLine;
            offset = selStartOffset;

            char [] s = lines [selStartLine] [0 .. selStartOffset];
            char [] e = lines [selEndLine] [selEndOffset .. lines [selEndLine].length];

            setLine (selStartLine, s ~ e);
            for (int c = selStartLine + 1; c <= selEndLine; c ++)
                delLine (selStartLine + 1);

            selClear ();
            view.caretChange ();
        }
    }

    void Return (View view)
    {
        char [] t, i;
        int indent = 0;

        DeleteSelection (view);

        with (view.document)
        {
            TabParams p = tabParams ();

            t = lines [line];

            indent = findIndent (view, line);
            lines [line] = lines [line] [0 .. offset];
            indent += hilite.indent (view.document, line) * p.indentSize;
            lines [line] = t;
            indent = imax (0, indent);

            i = new char [indent];
            i [] = " ";

            char [] r = p.reduceIndent (i ~ t [offset .. t.length]);

            addLine (line + 1, r);
            setLine (line, t [0 .. offset]);
            offset = p.reduceIndent (i).length;
            line ++;
            view.caretChange ();
        }
    }

    void Copy (View view)
    {
        with (view.document)
        {
            if (!selExists ())
                return;

            if (!view.clipboard.open ())
                return;

            try
            {
                char [] text;
                int length = 0;

                for (int c = selStartLine; c <= selEndLine; c ++)
                {
                    char [] l = lines [c];
                    int s = (c == selStartLine) ? selStartOffset : 0;
                    int e = (c == selEndLine) ? selEndOffset : l.length;

                    text ~= l [s .. e];
                    if (c != selEndLine)
                        text ~= "\n";
                }

                view.clipboard.empty ();
                view.clipboard.addText (text);
            }
            finally view.clipboard.close ();
        }
    }

    void Cut (View view)
    {
        view.run ("Copy");
        view.run ("DeleteSelection");
    }

    void Paste (View view)
    {
        with (view.document)
        {
            view.clipboard.open ();
            try
            {
                char [] [] split;
                char [] text;

                text = view.clipboard.getText ();
                if (text === null)
                    return;

                view.run ("DeleteSelection");
                split = digCommonFullSplitLines (text);
                
                char [] t = lines [line];
                char [] a = t [0 .. offset];
                char [] b = t [offset .. t.length];

                if (split.length == 1)
                {
                    setLine (line, a ~ split [0] ~ b);
                    offset = a.length + split [0].length;
                }
                else
                {
                    setLine (line, a ~ split [0]);

                    for (int c = 1; c < split.length - 1; c ++)
                        addLine (line + c, split [c]);

                    addLine (line + split.length - 1, split [split.length - 1] ~ b);

                    line += split.length - 1;
                    offset = split [split.length - 1].length;
                }

                view.caretChange ();
            }
            finally view.clipboard.close ();
        }
    }

    /* Indent this line or this selection. */
    void Indent (View view)
    {
        with (view.document)
        {
            TabParams p = tabParams ();

            if (selExists ())
            {
                char [] base = new char [p.indentSize];
                selStartOffset = 0;
                if (selEndOffset)
                    selEndLine ++;
                selEndOffset = 0;

                base [] = " ";
                for (int c = selStartLine; c < selEndLine; c ++)
                    setLine (c, p.reduceIndent (base ~ lines [c]));
            }
            else
            {
                char [] l = lines [line];
                char [] i;

                i.length = (offset + p.indentSize) / p.indentSize * p.indentSize - offset;
                i [0 .. i.length] = " ";

                setLine (line, p.reduceIndent (l [0 .. offset] ~ i ~ l [offset .. l.length]));
                offset += i.length;
            }

            view.caretChange ();
        }
    }

    /* Remove an indentation level from this line or this selection. */
    void Dedent (View view)
    {
        with (view.document)
        {
            int s = offset, e = offset + 1;
            TabParams p = tabParams ();

            if (selExists ())
            {
                selStartOffset = 0;
                if (selEndOffset)
                    selEndLine ++;
                selEndOffset = 0;
                s = selStartLine;
                e = selEndLine;
            }

            for (int c = s; c < e; c ++)
            {
                char [] line = p.expandIndent (lines [c]);

                for (int d; d < p.indentSize; d ++)
                {
                    if (!line.length || line [0] != ' ')
                        break;
                    line = line [1 .. line.length];
                    if (c == offset)
                        offset --;
                }

                setLine (c, p.reduceIndent (line));
            }
            
            view.caretChange ();
        }
    }

    void ScrollUp (View view)
    {
        view.document.lineOffset -= 1;
        view.paint ();
    }
    
    void ScrollDown (View view)
    {
        view.document.lineOffset += 1;
        view.paint ();
    }
    
    void ScrollCenter (View view)
    {
        view.document.lineOffset = view.document.line - view.height () / global.fontHeight;
        view.paint ();
    }
}
